#!/bin/bash

# Функция для настройки Docker на RPM-based системах
setup_docker_rpm() {
    echo "Обнаружена RPM-based система. Настройка Docker..."

    # Установка необходимых пакетов
    if command -v dnf &> /dev/null; then
        sudo dnf install -y dnf-plugins-core
    elif command -v yum &> /dev/null; then
        sudo yum install -y yum-utils
    fi

    # Установка Docker CE
    if command -v dnf &> /dev/null; then
        sudo dnf install -y docker-ce docker-ce-cli docker-compose-switch
    elif command -v yum &> /dev/null; then
        sudo yum install -y docker-ce docker-ce-cli docker-compose-switch
    fi

    # Добавление текущего пользователя в группу docker
    sudo usermod -aG docker $USER

    # Запуск и включение Docker в автозагрузку
    sudo systemctl enable docker
    sudo systemctl start docker

    echo "Docker успешно установлен и настроен на RPM-based системе."
}

# Функция кастомизации установки
setup_dir() {
    # Создание директорий
    sudo mkdir -p /opt/ray_project/{airflow-volume/dags,html,postgres-db-volume,trino/catalog,ssh_key,sql_migrations}

    # Копирование файлов
    sudo cp ./default.conf /opt/ray_project/
    sudo cp ./sql_migrations /opt/ray_project/

    # Настройка прав доступа
    sudo chown -R 999:999 /opt/ray_project/ssh_key
    sudo chown -R 50000:0 /opt/ray_project/airflow-volume
    sudo chown -R 999:999 /opt/ray_project/trino/catalog
    
    sudo chmod -R 765 /opt/ray_project/trino/catalog
    sudo chmod 775 /opt/ray_project/ssh_key
    sudo chmod +777 /opt/ray_project/airflow-volume/dags

    # Распаковка архива
    sudo tar xvf ./dist.tar -C /opt/ray_project/html/

    # Копирование миграций
    sudo cp -r ./sql_migrations/* /opt/ray_project/sql_migrations/

    # Получение внутренних IP-адресов
    internal_ips=$(ip addr show | grep 'inet ' | grep -v '127.0.0.1' | grep -v 'docker' | grep -v 'br-' | grep -v '172.17.' | grep -v '172.18.' | awk '{print $2}' | cut -d/ -f1)

    # Замена IP-адреса в файлах
    sudo grep -rl "172.22.41.231" /opt/ray_project/html/ | sudo xargs sed -i "s/172.22.41.231/$internal_ips/g"
}

# Функция для загрузки Docker образа из файла по маске
load_docker_image() {
    local image_mask=$1
    # Поиск файла, соответствующего маске
    local image_file=$(find . -name "$image_mask" -print -quit)

    if [[ -f $image_file ]]; then
        echo "Загрузка Docker образа из файла: $image_file"
        sg docker -c "docker load -i $image_file"
        if [[ $? -eq 0 ]]; then
            echo "Образ успешно загружен в Docker."
        else
            echo "Ошибка при загрузке образа."
            exit 1
        fi
    else
        echo "Файл образа, соответствующий маске $image_mask, не найден."
        exit 1
    fi
}

# Функция для проверки статуса контейнеров
check_health() {
    local container_name=$1
    local health_status
    health_status=$(sudo docker inspect --format='{{.State.Health.Status}}' "$container_name" 2>/dev/null)
    if [ "$health_status" == "healthy" ]; then
        return 0
    else
        return 1
    fi
}

# Функция погдоготвки файла .env
prepare_env_file() {
    local env_file=".env"
    sleep 5
    BACKEND_IMAGE_VER=$(sudo docker images --filter=reference='ray-gw' --format "{{.Tag}}")

    # Проверяем, существует ли переменная BACKEND_IMAGE_VER в файле .env
    if grep -q "^BACKEND_IMAGE_VER=" "$env_file"; then
        # Если строка существует, заменяем её значение на новое
        sed -i "s/^BACKEND_IMAGE_VER=.*/BACKEND_IMAGE_VER=${BACKEND_IMAGE_VER}/" "$env_file"
        echo "Значение BACKEND_IMAGE_VER обновлено в файле $env_file."
    else
        # Если строка отсутствует, добавляем её в конец файла
        echo "BACKEND_IMAGE_VER=${BACKEND_IMAGE_VER}" | tee -a "$env_file"
        echo "Строка BACKEND_IMAGE_VER добавлена в файл $env_file."
    fi
    # sed -i "s/172.22.41.231/$internal_ips/g" .env
    sed -i "s|^RAY_SSH_MASTER_HOST=.*|RAY_SSH_MASTER_HOST=developer@${internal_ips}|" "$env_file"
    sed -i "s|^VITE_APP_API_HOST=.*|VITE_APP_API_HOST=http://${internal_ips}:8082|" "$env_file"
    cp .env ./external_services/
    cp .env ./backend/
}

# Функция установки всех компонентов для rpm-based
setup_images_rpm() {
    # Запуск docker-compose для внешних сервисов
    sg docker -c "docker compose -f ./external_services/docker-compose.yml up -d"

    # Список компонентов для проверки статуса
    components=("postgres" "airflow-worker" "airflow-scheduler" "airflow-webserver" "airflow-triggerer")

    # Ожидание перехода всех компонентов в статус healthy
    for component in "${components[@]}"; do
        echo "Ожидание перехода $component в статус healthy..."
        start_time=$SECONDS
        while ! check_health "$component"; do
            elapsed_time=$((SECONDS - start_time))
            echo "Ожидание перехода $component в статус healthy... $elapsed_time сек"
            sleep 5
        done
        echo "$component теперь healthy."
    done

    # Подготовка базы данных для backend
    sudo docker exec -it postgres bash -c '
    # Проверка существования базы данных ray_gw и создание, если она не существует
    if ! psql -U airflow -d airflow -tc "SELECT 1 FROM pg_database WHERE datname = '\''ray_gw'\''" | grep -q 1; then
        psql -U airflow -d airflow -c "CREATE DATABASE ray_gw;"
    fi

    # Подключение к базе данных ray_gw и выполнение SQL-команд
    psql -U airflow -d ray_gw <<EOF
        DO \$\$
        BEGIN
            -- Проверка существования пользователя gw-user
            IF NOT EXISTS (SELECT 1 FROM pg_roles WHERE rolname = '\''gw-user'\'') THEN
                CREATE USER "gw-user" WITH PASSWORD '\''gw-passwd'\'';
                CREATE SCHEMA IF NOT EXISTS ray AUTHORIZATION "gw-user";
            END IF;

            -- Предоставление всех привилегий пользователю на базу данных ray_gw
            GRANT ALL PRIVILEGES ON DATABASE ray_gw TO "gw-user";
            GRANT ALL PRIVILEGES ON SCHEMA ray TO "gw-user";
            ALTER ROLE "gw-user" SUPERUSER CREATEDB CREATEROLE INHERIT LOGIN NOREPLICATION NOBYPASSRLS;
            -- Определение схемы по-умолчанию для пользователя gw-user
            ALTER USER "gw-user" SET search_path TO ray;
        END
        \$\$;
EOF
'
    # Установка компонент на основе образа ray-gw
    sg docker -c "docker compose -f ./backend/docker-compose.yml up -d"
    sleep 15
    sg docker -c "docker exec ray-gw python webservice/bin/create_super_user.py --username admin --full_name 'Супер Администратор' --password password"
    sg docker -c "docker exec ray-trino-monitor ssh-keygen -t rsa -b 4096 -f /home/appuser/.ssh/id_rsa -N \"\""
    sg docker -c "docker cp ray-trino-monitor:/home/appuser/.ssh/id_rsa.pub /tmp/id_rsa.pub"
    cat /tmp/id_rsa.pub >> ~/.ssh/authorized_keys
    chmod 600 ~/.ssh/authorized_keys
    rm /tmp/id_rsa.pub
}

# Функция для проверки наличия DAG и его активации
activate_dag() {
    # Адрес API Airflow
    AIRFLOW_API="http://$internal_ips:9080/api/v1/dags/trino"
    start_time=$SECONDS
    
    while true; do
        # Проверяем наличие DAG
        response=$(curl -s -o /dev/null -w "%{http_code}" "$AIRFLOW_API" "${HEADERS[@]}")

        if [ "$response" -eq 200 ]; then
            # Если DAG найден, активируем его
            curl -X PATCH "$AIRFLOW_API" "${HEADERS[@]}" -d '{"is_paused": false}'
            echo "DAG trino активирован."
            break
        else
            elapsed_time=$((SECONDS - start_time))
            echo "Ожидание готовности DAG trino... $elapsed_time сек"
            sleep 10
        fi
    done
}

# Заголовки для запроса
HEADERS=(
    -H 'Authorization: Basic YWlyZmxvdzphaXJmbG93'
    -H 'Content-Type: application/json'
)

# Установка Docker
setup_docker_rpm

# Кастомизация файлов и папок
setup_dir

# Загрузка необходимых образов по маске
load_docker_image "ray-gw.tar.gz"

# Подготовка файла окружения
prepare_env_file

# Установка компонентов
setup_images_rpm

# Активация DAG
activate_dag

# Проверка установки Docker
docker --version

echo "Перезагрузите систему или выполните 'newgrp docker' для применения изменений."
echo "Для входа используйте Яндекс Браузер версии не ниже 23 или Google Chrome"
echo "URL для входа: http://$internal_ips"